package presentation.util;

/**
 * @author Alberto Iachini, Concetta Balestra
 */

import java.util.Iterator;
import java.util.Set;
import java.util.Map.Entry;
import control.*;
import abstraction.Service;
import presentation.CustomTableModel;

import javax.swing.table.DefaultTableModel;

public class XmlTableOrchestratorFactory {
	
	private static final String HEADER_XML = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
	private static final String HREF_XSL = "<?xml-stylesheet href=\"";
	private static final String HREF_TYPE = "\" type=\"text/xsl\"?>\n\n";
	private static final String[] orchestratorColumnName = {"Target Service State","Available Services States","Action","Available Services"};
	private static final String[] simulationColumnName = {"Simulated Target State ","Is Simulated by (Community State)"};
	
	
	private String xmlOrchestrator = null;
	private String xmlSimulation = null;
	
	private CustomTableModel orchestratorModel = null;
	private CustomTableModel simulationModel = null;
	
	private Orchestrator orchestrator = null;
	
	public XmlTableOrchestratorFactory(Orchestrator orch)
	{
		orchestrator = orch;
	}
	
	/**
	 * Getter of orchestator xml
	 * @return the orchestrator in xml format
	 */
	public String getXmlOrchestrator() {
		if (xmlOrchestrator == null)
			orchestratorMapToString();
		return xmlOrchestrator;
	}

	/**
	 * Getter of simulation xml
	 * @return the simulation in xml format
	 */
	public String getXmlSimulation() {
		if (xmlSimulation == null)
			simulationToString();
		return xmlSimulation;
	}

	/**
	 * Return the table model that holds the orchestratot
	 * @return the orchestrator in table format
	 */
	public DefaultTableModel getOrchestratorModel() {
		if (orchestratorModel == null)
			orchestratorMapToString();
		return orchestratorModel;
	}

	/**
	 * Return the table model that holds the simulation
	 * @return the simulation in table format
	 */
	public DefaultTableModel getSimulationModel() {
		if (simulationModel == null)
			simulationToString();
		return simulationModel;
	}
	
	/**
	 * Return the style sheet for xml orchestrator 
	 * @return the style sheet
	 */
	public String getXslOrchestrator()
	{
		return XSL_ORCHESTRATOR;
	}
	/**
	 * Return the style sheet for xml simulation 
	 * @return the style sheet
	 */
	public String getXslSimulation()
	{
		return XSL_SIMULATION;
	}

	/**
	 * produces a string in xml format and a table
	 * containing all orchestrator result 
	 */
	private void orchestratorMapToString () {
		orchestratorModel = new CustomTableModel(orchestratorColumnName);
		Iterator<Entry<OrchestratorKey, Set<Service>>> it = orchestrator.getOrchestrator();
		StringBuilder xml = new StringBuilder();
		xml.append(HEADER_XML);
		xml.append(HREF_XSL + "orch.xsl" + HREF_TYPE);
		xml.append("<Orchestrator> \n");
		while (it.hasNext()) {
			Entry<OrchestratorKey, Set<Service>> set =(Entry<OrchestratorKey, Set<Service>>)it.next();
			OrchestratorKey key = (OrchestratorKey)set.getKey();
			SimulatedBy sim = key.getSimulatedBy();
			CState cs = sim.getCommunityState();
			String[] row = new String[4];
			
			// line
			xml.append("\t<OrchestratorLine>\n");
			
			//tag input
			xml.append("\t\t<Input>\n");
			
			
			// target state name
			xml.append("\t\t\t<TargetState name=\"" + sim.getTargetState() + "\"/>\n");
			row[0] = sim.getTargetState().getName();
			
			// service state name
			StringBuilder asn = new StringBuilder();
			int n = cs.getSize();
			for (int i=0; i<n; i++) {
				xml.append("\t\t\t<AvailableState name=\"" + cs.get(i) + "\"/>\n");
				asn.append(cs.get(i)).append(",");
			}
			asn.deleteCharAt(asn.length()-1);
			row[1]= asn.toString();
			
			// action
			xml.append("\t\t\t<Action name=\"" + key.getAction() + "\"/>\n");
			row[2] = key.getAction().getName();
			
			// end tag input
			xml.append("\t\t</Input>\n");
			
			// tag available services
			Set<Service> servs=(Set<Service>)set.getValue();
			xml.append("\t\t<AvailableServices>\n");
			Iterator<Service> iter = servs.iterator();
			StringBuilder svs= new StringBuilder();
			while (iter.hasNext()) {
				Service serv = (Service)iter.next();
				xml.append("\t\t\t<Service name =\"" + serv.getName() + "\"/>\n");
				svs.append(serv.getName()).append(",");
			}
			svs.deleteCharAt(svs.length()-1);
			row[3] = svs.toString();
			orchestratorModel.addRow(row);
			
			// end tag available services
			xml.append("\t\t</AvailableServices>\n");
			
			// end line
			xml.append("\t</OrchestratorLine>\n");
		}
		xml.append("</Orchestrator> \n");
		xmlOrchestrator = xml.toString();
	}
	
	/**
	 * produces a string in xml format and the table
	 * containing all simulation set  
	 */
	private void simulationToString () {
		simulationModel = new CustomTableModel(simulationColumnName);
		Iterator<SimulatedBy> it = orchestrator.getSimulationSet();
		StringBuilder xml = new StringBuilder();
		
		xml.append(HEADER_XML);
		xml.append(HREF_XSL + "sim.xsl"+ HREF_TYPE);
		xml.append("<SimulationSet> \n");
		while (it.hasNext()) {
			SimulatedBy set =(SimulatedBy)it.next();
			CState cs = set.getCommunityState();
			String[] row = new String[2];
			
			xml.append("\t<SimulatedStates>\n");
			xml.append("\t\t<TargetState name=\"" + set.getTargetState() + "\"/>\n");
			row[0] = set.getTargetState().getName();
			
			int n = cs.getSize();
			xml.append("\t\t<CommunityState name=\"");
			
			StringBuilder csm = new StringBuilder();
			for (int i=0; i<n; i++) {
				csm.append(cs.get(i));
			}
			xml.append(csm);
			row[1]= csm.toString();
			
			xml.append("\"/>\n");
			xml.append("\t</SimulatedStates>\n");
			
			simulationModel.addRow(row);
		}
		xml.append("</SimulationSet> \n");
		xmlSimulation = xml.toString();
	}
	
	private static final String XSL_ORCHESTRATOR ="<?xml version=\"1.0\"?>\n<xsl:stylesheet xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\" version=\"1.0\">\n<xsl:output method=\"html\"/>\n\n<xsl:template match=\"/Orchestrator\">\n\t<html>\n\t\t<body>\n\t\t\t<TABLE width=\"90%\" border=\"2\" bordercolor=\"#000000\">\n\t\t\t\t<CAPTION>\n\t\t\t\t\t<DIV align=\"center\"><font size=\"4\"><b>ORCHESTRATOR RESULT</b></font></DIV>\n\t\t\t\t\t<xsl:text>&#160;</xsl:text>\n\t\t\t\t</CAPTION>\n\t\t\t\t<THEAD>\n\t\t\t\t\t<TR>\n\t\t\t\t\t\t<TH><font size=\"4\">Target Service State</font></TH>\n\t\t\t\t\t\t<TH><font size=\"4\">Available Services States</font></TH>\n\t\t\t\t\t\t<TH><font size=\"4\">Action</font></TH>\n\t\t\t\t\t\t<TH><font size=\"4\">Available Services</font></TH>\n\t\t\t\t\t</TR>\n\t\t\t\t</THEAD>\n\t\t\t\t<TBODY>\n\t\t\t\t\t<xsl:apply-templates/>\n\t\t\t\t</TBODY>\n\t\t\t</TABLE>\n\t\t</body>\n\t</html>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine\">\n\t<TR>\n\t\t<xsl:apply-templates select=\"Input\"/>\n\t\t<TD>\n\t\t\t<xsl:apply-templates select=\"AvailableServices\"/>\n\t\t</TD>\n\t</TR>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/Input\">\n\t<xsl:apply-templates select=\"TargetState\"/>\n\t<TD>\n\t\t<DIV align=\"center\"><xsl:apply-templates select=\".//AvailableState\"/></DIV>\n\t\t\n\t</TD>\n\t<xsl:apply-templates select=\"Action\"/>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/Input/TargetState\">\n\t<TD><DIV align=\"center\"><xsl:value-of select=\"@name\"/></DIV></TD>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/Input/AvailableState\">\n\t<xsl:value-of select=\"@name\"/>,\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/Input/AvailableState[last()]\">\n\t<xsl:value-of select=\"@name\"/>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/Input/Action\">\n\t<TD><DIV align=\"center\"><xsl:value-of select=\"@name\"/></DIV></TD>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/AvailableServices\">\n\t<DIV align=\"center\"><xsl:apply-templates /></DIV>\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/AvailableServices/Service\">\n\t<xsl:value-of select=\"@name\"/>,\n</xsl:template>\n\n<xsl:template match=\"/Orchestrator/OrchestratorLine/AvailableServices/Service[last()]\">\n\t<xsl:value-of select=\"@name\"/>\n</xsl:template>\n\n</xsl:stylesheet>";
	private static final String XSL_SIMULATION ="<?xml version=\"1.0\"?>\n<xsl:stylesheet xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\" version=\"1.0\">\n<xsl:output method=\"html\"/>\n\n<xsl:template match=\"/SimulationSet\">\n\t<html>\n\t\t<body>\n\t\t\t<TABLE width=\"60%\" border=\"2\" bordercolor=\"#000000\">\n\t\t\t\t<CAPTION>\n\t\t\t\t\t<DIV align=\"CENTER\"><font size=\"4\"><b>COMPOSITION RESULT </b></font></DIV>\n\t\t\t\t\t<xsl:text>&#160;</xsl:text>\n\t\t\t\t</CAPTION>\n\t\t\t\t<THEAD>\n\t\t\t\t\t<TR>\n\t\t\t\t\t\t<TH>\n\t\t\t\t\t\t\t<font size=\"4\">Simulated Target State</font>\n\t\t\t\t\t\t</TH>\n\t\t\t\t\t\t<TH>\n\t\t\t\t\t\t\t<font size=\"4\">Is Simulated by (Community State)</font>\n\t\t\t\t\t\t</TH>\n\t\t\t\t\t</TR>\n\t\t\t\t</THEAD>\n\t\t\t\t<TBODY>\n\t\t\t\t\t<xsl:apply-templates/>\n\t\t\t\t</TBODY>\n\t\t\t</TABLE>\n\t\t</body>\n\t</html>\n</xsl:template>\n\n<xsl:template match=\"/SimulationSet/SimulatedStates\">\n\t<TR>\n\t\t<xsl:apply-templates/>\n\t</TR>\n</xsl:template>\n\n<xsl:template match=\"/SimulationSet/SimulatedStates/TargetState\">\n\t<TD>\n\t\t<DIV align=\"center\"><xsl:value-of select=\"@name\"/></DIV>\n\t</TD>\n</xsl:template>\n\n<xsl:template match=\"/SimulationSet/SimulatedStates/CommunityState\">\n\t<TD><DIV align=\"center\"><xsl:value-of select=\"@name\"/></DIV>\n\t</TD>\n</xsl:template>\n\n</xsl:stylesheet>";
}